/** @file   highscoretable.cpp
 * @brief   Implementation of HighscoreTable - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "highscoretable.h"
#include <fstream>
#include "filehasher.h"
#include "utils.h"
#include "www_assert.h"
using std::string;
using std::vector;
using std::ifstream;
using std::ofstream;
using std::istream;
using std::endl;

namespace WeWantWar {

///
/// Constructors, destructor and operators
/// ======================================

/** Constructor
 */
HighscoreTable::HighscoreTable( const string& filename ) :
  m_highscorefile( filename )
{
  this->openHighscoreFile();
}


/** Destructor
 */
HighscoreTable::~HighscoreTable()
{
}

/** Suggests the given entry
 */
void HighscoreTable::suggestEntry( int scores, const string& rName )
{
  int pos = this->getPosition( scores );
  if ( pos < 0 ) {
    return;
  }
  
  vector<Entry>::iterator it = m_entries.begin() + pos;
  Entry e;
  e.scores = scores;
  e.name = rName;
  m_entries.insert( it, e );
  
  m_entries.pop_back();
}



/** Saves the highscoretable
 */
int HighscoreTable::save() const
{
  ofstream fout( m_highscorefile.c_str() );
  if ( !fout ) {
    return -1;
  }
  
  fout << "# DO NOT EDIT THIS FILE !!" << endl;
  fout << "# VERY LIKELY YOU JUST BREAK IT AND" << endl;
  fout << "# CURRENT HIGHSCORES ARE LOST." << endl;
  fout << endl;
  fout << "<wewantwar_highscoretable>" << endl;
  
  // Write the entries
  for ( int i=0; i < m_entries.size(); i++ ) {
    // Replacing the spaces by '_'
    string tmpName = m_entries.at(i).name;
    for ( int k=0; k < tmpName.length(); k++ ) {
      if ( tmpName[k] == ' ' ) {
        tmpName[k] = '_';
      }
    }

    fout << "    <entry> score: " << m_entries.at(i).scores;
    fout << " name: " << tmpName << " </entry>" << endl;
  }
  
  fout << "</wewantwar_highscoretable>" << endl;
  fout << "[END]" << endl;
  fout.close();
  
  // Set the hashcode.
  eng2d::FileHasher hf( m_highscorefile );
  hf.set();
  return 0;
}

/** Tells the position where you got with given scores
 */
int HighscoreTable::getPosition( int score ) const
{
  for ( int i=0; i < m_entries.size(); i++ ) {
    if ( score > m_entries.at(i).scores ) {
      return i;
    }
  }
  return -1;
}



/** Returns the index'th entry
 */
HighscoreTable::Entry HighscoreTable::getEntry( int index ) const
{
  WWW_ASSERT( index >= 0 );
  WWW_ASSERT( index < m_entries.size() );
  return m_entries.at(index);
}



/** Returns the number of entries
 */
int HighscoreTable::numberOfEntries() const
{
  return m_entries.size();
}




///
/// Private or Protected methods
/// ============================

/** Creates the default highscoretable
 */
void HighscoreTable::createDefaultHighscores()
{
  m_entries.clear();
  Entry e;
  
  e.scores = 5000;
  e.name = "herr wood";
  m_entries.push_back( e );
  
  e.scores = 4500;
  e.name = "legoland";
  m_entries.push_back( e );
  
  e.scores = 4000;
  e.name = "angelo";
  m_entries.push_back( e );
  
  e.scores = 3500;
  e.name = "predator";
  m_entries.push_back( e );
  
  e.scores = 3000;
  e.name = "arnold";
  m_entries.push_back( e );
  
  e.scores = 1500;
  e.name = "we";
  m_entries.push_back( e );
  
  e.scores = 1000;
  e.name = "want";
  m_entries.push_back( e );
  
  e.scores = 500;
  e.name = "war";
  m_entries.push_back( e );
}



/** Opens the highscorefile
 */
void HighscoreTable::openHighscoreFile()
{
  ifstream fin( m_highscorefile.c_str() );
  if ( !fin ) {
    this->createDefaultHighscores();
    return;
  }
  
  if ( Utils::searchForString( fin, "<wewantwar_highscoretable>" ) == true ) {
    // No opening tag, so the file is invalid. We create the default table.
    fin.close();
    this->createDefaultHighscores();
    return;
  }

  // Read the entries.
  while ( true ) {
    if ( fin.eof() == true ) {
      // EOF encountered. Invalid file. We create the default table.
      fin.close();
      this->createDefaultHighscores();
      return;
    }
    
    string tmp;
    fin >> tmp;
    if ( tmp == "</wewantwar_highscoretable>" ) {
      break;
      
    } else if ( tmp == "<entry>" ) {
      int ret = this->readEntry( fin );
      if ( ret != 0 ) {
        fin.close();
        this->createDefaultHighscores();
        return;
      }
      
    } else if ( tmp == "#" ) {
      fin.ignore( 4096, '\n' );
      
    }
  }
  fin.close();
  
  // If the hashcode is invalid we reject the highscoretable we've just
  // read.
  eng2d::FileHasher hf( m_highscorefile );
  if ( hf.check() == true ) {
    this->createDefaultHighscores();
  }
  return;
}



/** Reads the single entry
 */
int HighscoreTable::readEntry( istream& rIn )
{
  Entry e;
  e.scores = 0;
  e.name = "";
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "</entry>" ) {
      m_entries.push_back( e );
      return 0;
      
    } else if ( tmp == "score:" ) {
      rIn >> e.scores;
      
    } else if ( tmp == "name:" ) {
      string n;
      rIn >> n;
      
      // Decode the spaces
      for ( int i=0; i < n.length(); i++ ) {
        if ( n[i] == '_' ) {
          n[i] = ' ';
        }
      }
      e.name = n;
    }
  }
  return 0;
}


} // end of namespace
